#!/usr/bin/env python3
"""
麦喵 API Python SDK 示例
演示如何使用麦喵开放 API 进行登录、创建卡组和卡片等操作
"""

import requests
import json

BASE_URL = "https://fc-mp-7c19cdfb-aa09-46b8-af6b-f7b52e3b3d7b.next.bspapp.com/open-api"
APP_ID = "__UNI__235AE90"


class MemocardAPI:
    """麦喵 API 客户端"""
    
    def __init__(self, app_id=APP_ID):
        self.base_url = BASE_URL
        self.app_id = app_id
        self.token = None
    
    def _headers(self, auth=False):
        """生成请求头"""
        headers = {
            "app-id": self.app_id,
            "Content-Type": "application/json"
        }
        if auth and self.token:
            headers["uni-id-token"] = self.token
        return headers
    
    def login(self, mobile, password):
        """用户登录"""
        url = f"{self.base_url}/login"
        data = {
            "mobile": mobile,
            "password": password
        }
        response = requests.post(url, headers=self._headers(), json=data)
        result = response.json()
        
        if result.get("errCode") == 0:
            self.token = result["data"]["token"]
            print(f"✅ 登录成功，用户ID: {result['data']['uid']}")
        else:
            print(f"❌ 登录失败: {result.get('errMsg')}")
        
        return result
    
    def get_current_user(self):
        """获取当前用户信息"""
        url = f"{self.base_url}/getCurrentUser"
        response = requests.post(url, headers=self._headers(auth=True))
        return response.json()
    
    def create_card_group(self, title, description="", tags=None, is_public=1, language=None):
        """创建卡组"""
        url = f"{self.base_url}/createCardGroup"
        data = {
            "title": title,
            "description": description,
            "is_public": is_public
        }
        if tags:
            data["tags"] = tags
        if language:
            data["language"] = language
        
        response = requests.post(url, headers=self._headers(auth=True), json=data)
        result = response.json()
        
        if result.get("errCode") == 0:
            print(f"✅ 卡组创建成功，ID: {result['data']['id']}")
        else:
            print(f"❌ 创建失败: {result.get('errMsg')}")
        
        return result
    
    def get_card_group_token(self, card_group_id):
        """获取卡组 Token"""
        url = f"{self.base_url}/getCardGroupOpenApiToken"
        data = {"cardGroupId": card_group_id}
        response = requests.post(url, headers=self._headers(auth=True), json=data)
        return response.json()
    
    def create_card(self, card_group_token, front, back, imgs=None, front_note="", back_note=""):
        """创建卡片"""
        url = f"{self.base_url}/createCard"
        data = {
            "cardGroupToken": card_group_token,
            "front": front,
            "back": back,
            "frontNote": front_note,
            "backNote": back_note
        }
        if imgs:
            data["imgs"] = imgs
        
        response = requests.post(url, headers=self._headers(), json=data)
        result = response.json()
        
        if result.get("errCode") == 0:
            print(f"✅ 卡片创建成功，ID: {result['data']['id']}")
        else:
            print(f"❌ 创建失败: {result.get('errMsg')}")
        
        return result
    
    def get_all_cards(self, card_group_id):
        """获取卡组所有卡片"""
        url = f"{self.base_url}/getAllCard"
        data = {"cardGroupId": card_group_id}
        response = requests.post(url, headers=self._headers(auth=True), json=data)
        return response.json()
    
    def get_unfamiliar_cards(self, card_group_id, only_is_show=True, only_show_today=False):
        """获取不熟悉卡片列表"""
        url = f"{self.base_url}/getAllUnfamiliarCard"
        data = {
            "cardGroupId": card_group_id,
            "onlyIsShow": only_is_show,
            "onlyShowToday": only_show_today
        }
        response = requests.post(url, headers=self._headers(auth=True), json=data)
        return response.json()


def main():
    """使用示例"""
    # 初始化客户端
    api = MemocardAPI()
    
    # 1. 登录
    # result = api.login("13800138000", "your-password")
    
    # 2. 获取用户信息
    # user_info = api.get_current_user()
    # print(json.dumps(user_info, indent=2, ensure_ascii=False))
    
    # 3. 创建卡组
    # result = api.create_card_group(
    #     title="英语基础词汇",
    #     description="适合初学者的英语词汇",
    #     tags=["英语", "基础"],
    #     language={"front": "en", "back": "zh"}
    # )
    
    # 4. 获取卡组 Token
    # token_result = api.get_card_group_token("your-card-group-id")
    # card_group_token = token_result["data"]["token"]
    
    # 5. 创建卡片
    # api.create_card(
    #     card_group_token=card_group_token,
    #     front="Hello",
    #     back="你好",
    #     front_note="英语问候语",
    #     back_note="中文翻译"
    # )
    
    print("请取消注释相应的代码行来测试 API 功能")


if __name__ == "__main__":
    main()
